<!DOCTYPE html>
<html>
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<title>Encarte Digital 1-4-1</title>
<link href="https://fonts.googleapis.com/css2?family=Roboto:wght@400;700;900&display=swap" rel="stylesheet">
<style type="text/css">
    /* --- RESET & BASE --- */
    * { margin: 0; padding: 0; box-sizing: border-box; }
    
    body, html {
        width: 100%; height: 100%;
        background-color: #121212; /* Fundo Dark Moderno */
        font-family: 'Roboto', 'Arial', sans-serif;
        overflow: hidden;
    }

    /* --- TOPO --- */
    #topo {
        width: 100%; height: 13vh;
        background: #d32f2f; /* Vermelho Institucional */
        border-bottom: 0.5vh solid #ffc107;
        display: flex; align-items: center; justify-content: center;
        box-shadow: 0 1vh 2vh rgba(0,0,0,0.5);
        position: relative; z-index: 10;
    }
    #topo h1 {
        font-size: 6vh; color: #fff;
        text-transform: uppercase; margin: 0;
        font-weight: 900; letter-spacing: 0.2vw;
    }

    /* --- CONTAINER PRINCIPAL --- */
    #main-container {
        position: absolute; top: 13vh; left: 0;
        width: 100%; height: 87vh;
        padding: 2vh 1.5vw;
        display: flex; /* Flexbox para dividir as 3 colunas principais */
        gap: 1.5vw;
    }

    /* ESTRUTURA DAS COLUNAS */
    .col-lateral { width: 25%; height: 100%; }
    
    /* O CENTRO AGORA É UM GRID 2x2 PARA FICAR BONITO */
    .col-central { 
        width: 50%; height: 100%; 
        display: grid;
        grid-template-columns: 1fr 1fr; /* 2 Colunas */
        grid-template-rows: 1fr 1fr;    /* 2 Linhas */
        gap: 1.5vh 1.5vw;               /* Espaço entre os cards */
    }

    /* --- CARD GERAL (Branco) --- */
    .card {
        background: white;
        border-radius: 2vh;
        overflow: hidden;
        display: flex; flex-direction: column;
        box-shadow: 0 1vh 2vh rgba(0,0,0,0.3);
        position: relative;
        animation: fadeIn 0.8s ease;
    }

    /* --- ESTILO DOS CARDS LATERAIS (Grandes) --- */
    .card-lat { height: 100%; border: 0.4vh solid #d32f2f; }
    
    .lat-tag {
        position: absolute; top: 0; right: 0;
        background: #ffc107; color: #d32f2f;
        padding: 0.5vh 2vw; font-weight: 900; font-size: 2vh;
        border-bottom-left-radius: 2vh;
        z-index: 5;
    }

    .lat-img-area {
        height: 55%; width: 100%;
        display: flex; align-items: center; justify-content: center;
        padding: 2vh;
    }
    .lat-img-area img {
        max-height: 100%; max-width: 95%;
        object-fit: contain;
    }

    .lat-info-area {
        height: 45%; width: 100%;
        background: #f5f5f5;
        display: flex; flex-direction: column;
        justify-content: center; align-items: center;
        padding: 1vh; text-align: center;
        border-top: 1px solid #ddd;
    }
    
    .lat-nome {
        font-size: 3.5vh; color: #333; font-weight: 700;
        line-height: 1.1; margin-bottom: 1vh;
        text-transform: uppercase;
        display: -webkit-box; -webkit-line-clamp: 2; -webkit-box-orient: vertical; overflow: hidden;
    }

    .lat-preco {
        font-size: 9vh; color: #d32f2f; font-weight: 900; letter-spacing: -0.2vh;
    }
    .lat-rs { font-size: 3vh; color: #d32f2f; font-weight: bold; margin-right: 5px; vertical-align: top; }
    .lat-un { font-size: 2vh; color: #777; font-weight: bold; }


    /* --- ESTILO DOS CARDS CENTRAIS (Menores) --- */
    .card-cen { width: 100%; height: 100%; border: 1px solid #ddd; }
    
    .cen-img-area {
        height: 50%; width: 100%;
        display: flex; align-items: center; justify-content: center;
        padding: 1vh;
    }
    .cen-img-area img {
        max-height: 100%; max-width: 90%;
        object-fit: contain;
    }

    .cen-info-area {
        height: 50%; width: 100%;
        padding: 0.5vh 1vh;
        text-align: center;
        display: flex; flex-direction: column; justify-content: center;
    }

    .cen-nome {
        font-size: 2.2vh; color: #333; font-weight: 700;
        margin-bottom: 0.5vh; text-transform: uppercase;
        white-space: nowrap; overflow: hidden; text-overflow: ellipsis;
    }

    .cen-preco { font-size: 6vh; color: #d32f2f; font-weight: 900; line-height: 1; }
    .cen-rs { font-size: 2vh; color: #555; font-weight: bold; vertical-align: top; }
    .cen-un { font-size: 1.5vh; color: #999; }

    /* --- UTILITÁRIOS --- */
    #status-dot { position: fixed; bottom: 10px; right: 10px; width: 12px; height: 12px; border-radius: 50%; background: #555; z-index: 999; border: 2px solid #fff; }
    @keyframes fadeIn { from { opacity: 0; transform: translateY(10px); } to { opacity: 1; transform: translateY(0); } }

</style>
</head>
<body>

    <div id="topo">
        <h1>Ofertas Especiais</h1>
    </div>

    <div id="main-container">
        <div class="col-lateral" id="col-esquerda"></div>
        
        <div class="col-central" id="col-centro"></div>
        
        <div class="col-lateral" id="col-direita"></div>
    </div>

    <div id="status-dot"></div>

    <script type="text/javascript">
        // --- CONFIGURAÇÃO ---
        var API_URL = 'https://mgaplay.com.br/ofertas/destaque-mercado.php'; 
        var placeholder = 'https://cdn-icons-png.flaticon.com/512/263/263142.png';
        
        var TEMPO_TROCA = 10000; // Tempo entre páginas
        var ITENS_POR_PAGINA = 6; // 1 Esq + 4 Centro + 1 Dir
        var TEMPO_ATUALIZACAO = 60000; // Buscar novos dados a cada 1 min
        
        var listaGlobal = [];
        var indiceGlobal = 0;

        // --- FUNÇÕES AUXILIARES ---
        function setStatus(cor) { document.getElementById('status-dot').style.background = cor; }

        function formatarMoeda(valor) {
            if (!valor) return "0,00";
            let v = String(valor).replace(',', '.');
            let floatV = parseFloat(v);
            if (isNaN(floatV)) return valor; 
            return floatV.toFixed(2).replace('.', ',');
        }

        // --- CACHE OFFLINE ---
        function salvarCache(dados) { try { localStorage.setItem('mga_141_new', JSON.stringify(dados)); } catch(e){} }
        function carregarCache() {
            try {
                var c = localStorage.getItem('mga_141_new');
                if (c) {
                    var d = JSON.parse(c);
                    if (d.length > 0) { listaGlobal = d; setStatus('blue'); renderizar(); return true; }
                }
            } catch(e) {}
            return false;
        }

        // --- BUSCA DADOS ---
        function buscarDados() {
            var urlParams = new URLSearchParams(window.location.search);
            var uid = urlParams.get('uid') || urlParams.get('loja') || 1;

            var xhr = new XMLHttpRequest();
            var url = API_URL + '?uid=' + uid + '&t=' + new Date().getTime();
            
            xhr.open("GET", url, true);
            xhr.timeout = 5000;

            xhr.onreadystatechange = function() {
                if (xhr.readyState == 4) {
                    if (xhr.status == 200) {
                        try {
                            var dados = JSON.parse(xhr.responseText);
                            if (!Array.isArray(dados)) dados = [dados];
                            
                            if (dados.length > 0) {
                                listaGlobal = dados;
                                salvarCache(dados);
                                setStatus('lime');
                                renderizar(); // Atualiza se chegou dados novos
                            } else {
                                if(listaGlobal.length === 0) carregarCache();
                            }
                        } catch (e) { if(listaGlobal.length === 0) carregarCache(); }
                    } else { if(listaGlobal.length === 0) carregarCache(); }
                }
            };
            xhr.onerror = function() { if(listaGlobal.length === 0) carregarCache(); };
            xhr.send();
        }

        // --- HTML TEMPLATES ---
        function getHtmlLateral(item) {
            // Se não tiver item, renderiza vazio transparente
            if(!item) return '<div style="width:100%;height:100%;"></div>';
            
            var img = (item.imagem && item.imagem.length > 5) ? item.imagem : placeholder;
            return `
            <div class="card card-lat">
                <div class="lat-tag">OFERTA</div>
                <div class="lat-img-area">
                    <img src="${img}" onerror="this.src='${placeholder}'">
                </div>
                <div class="lat-info-area">
                    <div class="lat-nome">${item.produto}</div>
                    <div>
                        <span class="lat-rs">R$</span>
                        <span class="lat-preco">${formatarMoeda(item.preco)}</span>
                        <div class="lat-un">/${item.unidade || 'un'}</div>
                    </div>
                </div>
            </div>`;
        }

        function getHtmlCentral(item) {
            if(!item) return '<div style="width:100%;height:100%;"></div>';
            
            var img = (item.imagem && item.imagem.length > 5) ? item.imagem : placeholder;
            return `
            <div class="card card-cen">
                <div class="cen-img-area">
                    <img src="${img}" onerror="this.src='${placeholder}'">
                </div>
                <div class="cen-info-area">
                    <div class="cen-nome">${item.produto}</div>
                    <div>
                        <span class="cen-rs">R$</span>
                        <span class="cen-preco">${formatarMoeda(item.preco)}</span>
                    </div>
                    <div class="cen-un">/${item.unidade || 'un'}</div>
                </div>
            </div>`;
        }

        // --- RENDERIZAÇÃO ---
        function renderizar() {
            var colEsq = document.getElementById('col-esquerda');
            var colCen = document.getElementById('col-centro');
            var colDir = document.getElementById('col-direita');
            
            if (listaGlobal.length === 0) return;

            // Limpa containers
            colEsq.innerHTML = '';
            colCen.innerHTML = '';
            colDir.innerHTML = '';

            // Preenche os 6 slots (1 Esq, 4 Centro, 1 Dir)
            for (var i = 0; i < ITENS_POR_PAGINA; i++) {
                var index = (indiceGlobal + i) % listaGlobal.length;
                var item = listaGlobal[index];

                if (i === 0) {
                    // Item 1: Esquerda
                    colEsq.innerHTML = getHtmlLateral(item);
                } else if (i >= 1 && i <= 4) {
                    // Itens 2, 3, 4, 5: Centro (Grid)
                    colCen.innerHTML += getHtmlCentral(item);
                } else if (i === 5) {
                    // Item 6: Direita
                    colDir.innerHTML = getHtmlLateral(item);
                }
            }

            // Incrementa índice para a próxima página
            indiceGlobal += ITENS_POR_PAGINA;
        }

        // --- INICIALIZAÇÃO ---
        window.onload = function() {
            carregarCache(); // Tenta carregar cache instantaneamente
            buscarDados();   // Busca dados novos
            
            // Loop de troca de produtos
            setInterval(function() { 
                if(listaGlobal.length > 0) renderizar(); 
            }, TEMPO_TROCA);
            
            // Loop de atualização de dados (Rede)
            setInterval(buscarDados, TEMPO_ATUALIZACAO);
        };
    </script>
</body>
</html>